function N = trigger (img,ref)
% N = trigger (img,ref)
%
% Triggering function for microfluic real-time microsphere sorter.
% Triggering can occur only if the maximum intensity of the 3x3
% median-filtered image is more than 500 counts above a reference intensity
% value. The microsphere sorter uses the mean background intensity from the
% first image of each sorting sequence as the reference intensity value. The
% function returns the number of regions detected by Otsu's thresholding,
% but the microsphere sorter uses the output as a a binary trigger only.
%
% In:
%  img     16-bit grayscale image
%  ref     reference intensity
%
% Out:
%  N       number of beads in the image
%
% Antti Niemist 2009/12/15

% Use appropriate data type
img = uint16(img);

% Filter the image with a 3x3 median filter to remove noisy spikes
img = medfilt2(img,'symmetric');

% No trigger if the maximum intensity is not more than 500 counts above the
% reference intensity value
if max(img(:))-ref < 500

  N = 0;

% Else trigger if bright objects larger than 50 pixels in size are found
else

  % Adjust the image such that the pixel values span [0,1]
  datatype = 'uint16';
  img = double(img)/double(intmax(datatype));
  img = imadjust(img,[min(img(:)),max(img(:))]);

  % Threshold the image with a threshold found by Otsu's method
  T = graythresh(img);
  bw = im2bw(img,T);

  % Remove small foreground objects
  minbeadsize = 50;
  bw = bwareaopen(bw,minbeadsize);
  
  % Count the number of objects N, triggers if N is above zero
  [dummy,N] = bwlabel(bw);

end
